//-----------------------------------------------------------------------------
// The Potion : A Small Adventure by Peter Karlsson, ported by Anara          |
//-----------------------------------------------------------------------------
#include <CyWin.h>
#define BOOL char
#define TRUE 1
#define FALSE 0
#define OBJECTS 21                                                 // Constants
#define INVENTORY 2
#define MAPSIZE 30
#define EXIT_NORTH 0x01
#define EXIT_SOUTH 0x02
#define EXIT_WEST  0x04
#define EXIT_EAST  0x08
#define SCREENWIDTH 31
#define VERBS 29
#define NOUNS 22
#define V_N 1
#define V_S 2
#define V_E 3
#define V_W 4
#define V_I 5
#define V_TAKE 6
#define V_LIFT 7
#define V_READ 8
#define V_TALKTO 9
#define V_UNLOCK 10
#define V_TURN 11
#define V_HIT 12
#define V_KILL 13
#define V_POUR 14
#define V_DRINK 15
#define V_THROW 16
#define V_LOOK 17
#define V_DROP 18
#define V_LICENSE 19
#define V_QUIT 20
#define O_STONE_PILL 0
#define O_STONE_1 1
#define O_STONE_2 2
#define O_STONE_3 3
#define O_STONE_MISSION 4
#define O_STONE_5 5
#define O_STONE_6 6
#define O_STONE_7 7
#define O_STONE_8 8
#define O_LAST_STONE O_STONE_8
#define O_PILL 9
#define O_SQUIRREL 10
#define O_COIN 11
#define O_COTTAGE 12
#define O_SWITCH 13
#define O_HUBCAP 14
#define O_KEY 15
#define O_SCIENTIST 16
#define O_ELIXIR 17
#define O_BOOK 18
#define O_DOOR 19
#define O_COKE 20
#define GAME_CONTINUES 0
#define GAME_LOST 1
#define GAME_WON 2

char tmp[64];         // << REMEMBER TO OPTIMIZE
char *input;
const char *object_string;
int screenwidth;
char move(char verb);
char inventory(void);
static const struct verb_s
{
  const char len;
  const char *s;
  char verbnum;
} verbs[VERBS] =
{
  { 1, "n", V_N },
  { 1, "s", V_S },
  { 1, "e", V_E },
  { 1, "w", V_W },
  { 1, "i", V_I },
  { 4, "take", V_TAKE },
  { 3, "get", V_TAKE }, //was 7, "pickup", V_TAKE }
  { 4, "lift", V_LIFT },
  { 4, "read", V_READ },
  { 7, "talk to", V_TALKTO },
  { 6, "unlock", V_UNLOCK },
  { 4, "open", V_UNLOCK },
  { 4, "turn", V_TURN },
  { 3, "hit", V_HIT },
  { 5, "punch", V_HIT },
  { 4, "kick", V_HIT },
  { 4, "kill", V_KILL },
  { 4, "pour", V_POUR },
  { 5, "empty", V_POUR },
  { 5, "drink", V_DRINK },
  { 5, "throw", V_THROW },
  { 7, "look at", V_LOOK },
  { 4, "look", V_LOOK },
  { 7, "examine", V_LOOK },
  { 4, "drop", V_DROP },
  { 7, "license", V_LICENSE },
  { 7, "licence", V_LICENSE },
  { 4, "quit", V_QUIT },
  { 4, "exit", V_QUIT },
};

int InitCon();                                        // Custom Port Prototypes
void Scroll(void);
void Clrline(void);
int puts(char *);
char *gets(char *);
int getchar(void);
void initgame(void);
void intro(void);
char take(char object);
char lift(char object);
char read(char object);
char talkto(char object);
char unlock(char object);
char turn(char object);
char hit(char object);
char kill(void);
char pour(char object);
char drink(char object);
char throw(char object);
char look(char object);
char drop(char object);
char license(void);
char quit(void);
char getinput(void);
void printobject(char);
char objectfromword(const char *, BOOL checkinventory);
BOOL addinventory(char object);
BOOL dropinventory(char object);
BOOL movable(char object);
void putstring(const char *s);
char perform(char verb);
void printroom(void);

struct Message* ptr_msg;                                             // Structs
struct module_t main_module;
struct gamedata_s
{
  char room; /* 1 based */
  char inventory[INVENTORY];
  char objects[OBJECTS];
  BOOL pilleaten;
  BOOL seenscientist;
  char map[MAPSIZE];
  char roadvisit;
  char bothbottles;
};
struct gamedata_s *gamedata; // = NULL;

//-----------------------------------------------------------------------------
// MAIN - Program Entry Point                                                 |
//-----------------------------------------------------------------------------
long main(int argc, char* argv[], bool start)
{
char exit;
char oldroom;
const char *s;
init_module(&main_module);                                 // Initialize Module
InitCon();                                                // Initialize Console
  initgame();
  intro();
  oldroom = 0;
  do
  {
    char verb;
    if (oldroom != gamedata->room)
    {
      printroom();
      oldroom = gamedata->room;
    }
    verb = getinput();
    exit = perform(verb);
  } while (GAME_CONTINUES == exit);
  switch (exit)
  {
    case GAME_LOST:
      beep(BEEP_ERROR);
      s = "GAME OVER!";
      break;
    case GAME_WON:
      beep(BEEP_INFO);
      s = "Congratulations!";
      break;
  }
  puts(s);
  free(gamedata);
  free(input);
  sleep(4000);
return 0;
}

static const char defaultrooms[OBJECTS] =
{1, 2, 3, 7, 8, 9, 13, 14, 15, -1, 9, 18, 16, -1, 24, 28, 19, 25, 26, 20, -1};

static const char defaultmap[MAPSIZE] =
{
                     EXIT_SOUTH |             EXIT_EAST /*  1 */,
                                  EXIT_WEST | EXIT_EAST /*  2 */,
                     EXIT_SOUTH | EXIT_WEST             /*  3 */,
                     EXIT_SOUTH | EXIT_WEST             /*  4 */,
                     EXIT_SOUTH |             EXIT_EAST /*  5 */,
                                  EXIT_WEST             /*  6 */,
        EXIT_NORTH | EXIT_SOUTH                         /*  7 */,
                     EXIT_SOUTH                         /*  8 */,
        EXIT_NORTH | EXIT_SOUTH                         /*  9 */,
        EXIT_NORTH | EXIT_SOUTH                         /* 10 */,
        EXIT_NORTH | EXIT_SOUTH | EXIT_WEST | EXIT_EAST /* 11 */,
                     EXIT_SOUTH | EXIT_WEST             /* 12 */,
        EXIT_NORTH |                          EXIT_EAST /* 13 */,
        EXIT_NORTH |              EXIT_WEST             /* 14 */,
        EXIT_NORTH                                      /* 15 */,
        EXIT_NORTH |                          EXIT_EAST /* 16 */,
        EXIT_NORTH | EXIT_SOUTH | EXIT_WEST | EXIT_EAST /* 17 */,
        EXIT_NORTH | EXIT_SOUTH | EXIT_WEST             /* 18 */,
                     EXIT_SOUTH |             EXIT_EAST /* 19 */,
                     EXIT_SOUTH |             EXIT_EAST /* 20 */,
                     EXIT_SOUTH | EXIT_WEST | EXIT_EAST /* 21 */,
                                              EXIT_EAST /* 22 */,
        EXIT_NORTH | EXIT_SOUTH | EXIT_WEST | EXIT_EAST /* 23 */,
        EXIT_NORTH | EXIT_SOUTH | EXIT_WEST             /* 24 */,
        EXIT_NORTH                                      /* 25 */,
        EXIT_NORTH |                          EXIT_EAST /* 26 */,
        EXIT_NORTH |              EXIT_WEST             /* 27 */,
                                              EXIT_EAST /* 28 */,
        EXIT_NORTH |              EXIT_WEST | EXIT_EAST /* 29 */,
                                  EXIT_WEST             /* 30 */
};

void initgame(void)
{
char i;
  gamedata = malloc(sizeof (struct gamedata_s));
  input = malloc(1024);
  for (i = 0; i < INVENTORY; ++ i)
    gamedata->inventory[i] = -1;
  gamedata->room = 15;
  gamedata->pilleaten = 0;
  gamedata->seenscientist = 0;
  gamedata->roadvisit = 0;
  memcpy(gamedata->objects, defaultrooms, OBJECTS);
  memcpy(gamedata->map, defaultmap, MAPSIZE);
}

void intro(void)
{
  puts("");
  putstring("  ||   The Potion\n (_)  A simple adventure");
  sleep(2000);
  putstring("\n Copyright 2002-2004\n by Peter Karlsson.\n ported by Anara\n");
  puts("");
  sleep(3000);
}

void cannotdothat(void)
{
  puts("You cannot do that!");
}

void nothing(void)
{
  puts("Nothing special.");
}

char move(char verb)
{
char bitmap;
char offset;
  /* Seen scientist? */
  if (gamedata->seenscientist)
  {
    puts("The scientist captures you.");
    return GAME_LOST;
  }
  /* FIXME: Crossing autobahn? */
  /* Check which movement is requested */
  switch (verb)
  {
    case V_N: bitmap = EXIT_NORTH; offset = -6; break;
    case V_S: bitmap = EXIT_SOUTH; offset =  6; break;
    case V_E: bitmap = EXIT_EAST;  offset =  1; break;
    case V_W: bitmap = EXIT_WEST;  offset = -1; break;
  }
  /* Check if movement is allowed in that direction */
  if (gamedata->map[gamedata->room - 1] & bitmap)
  {
    gamedata->room += offset;
  }
  else
  {
    puts("You cannot go that way.");
    return GAME_CONTINUES;
  }
  /* Check if we are carrying both bottles */
  if ((O_ELIXIR == gamedata->inventory[0] && O_COKE == gamedata->inventory[1])
  || (O_ELIXIR == gamedata->inventory[1] && O_COKE == gamedata->inventory[0]))
  {
    switch (++ gamedata->bothbottles)
    {
      case 1:
        putstring("The potion is starting to heat up.");
        break;
      case 5:
        beep(BEEP_QUESTION);
        putstring("A reaction between the potion and the Coke is happening.");
        break;
      case 10:
        beep(BEEP_QUESTION);
        putstring("Get rid of one of the bottles before it explodes!");
        break;
      case 12:
        putstring("The potion explodes in your face!");
        return GAME_LOST;
    }
  }
  else
  {
  gamedata->bothbottles = 0;                                   // Reset counter
  }
return GAME_CONTINUES;
}

char inventory(void)
{
BOOL hasanything;
char i;
char *inventory;
  hasanything = FALSE;
  inventory = gamedata->inventory;
  puts("You are carrying:");
  for (i = INVENTORY; i; -- i, ++ inventory)
    if (*inventory != -1)
    {
      printobject(*inventory);
      hasanything = TRUE;
    }
    if (!hasanything)
      nothing();
return GAME_CONTINUES;
}

char take(char object)
{
  switch (object)
  {
    case O_PILL:
      /* Something special happens when we take the pill */
      gamedata->pilleaten = TRUE;
      gamedata->objects[O_PILL] = -1;
      beep(BEEP_OK);
      puts("You feel odd but understand.");
      break;
    case O_STONE_PILL:
      /* Finds pill, if it's not taken already */
      if (!gamedata->pilleaten && -1 == gamedata->objects[O_PILL])
      {
        putstring("You found a small pill beneath it.");
        gamedata->objects[O_PILL] = gamedata->room;
        break;
      }
      /* Else fall through */
    default:
      if (movable(object))
      {
        if (addinventory(object))
          puts("Taken.");
      }
      else
      {
        cannotdothat();
      }
    break;
  }
return GAME_CONTINUES;
}

char lift(char object)
{
  switch (object)
  {
    case O_STONE_PILL:
      if (!gamedata->pilleaten)
      {
        take(object);
        break;
      }
      /* else fall through */
    default:
      nothing();
      break;
  }
return GAME_CONTINUES;
}

char read(char object)
{
  switch (object)
  {
    case O_STONE_MISSION:
      beep(BEEP_OK);
      putstring("It says:\n\"Save us from the evil scientist. Mix his evil drug with salt water. That is your mission.\"");
      break;
    default:
      nothing();
  }
return GAME_CONTINUES;
}

char talkto(char object)
{
  switch (object)
  {
    case O_SCIENTIST:
      beep(BEEP_INFO);
      putstring("The laughing scientist says:\n\"You'll die with the rest!\"");
      break;
    case O_SQUIRREL:
      if (gamedata->pilleaten)
      {
      	beep( BEEP_QUESTION );
        putstring("The squirrel says:\n\"You can now exit the forest\"");
        gamedata->map[3 - 1] |= EXIT_EAST;
        break;
      }
    default:                        
      cannotdothat();
  }
return GAME_CONTINUES;
}

char unlock(char object)
{
BOOL havekey;
char i;
  havekey = FALSE;
  for (i = 0; i < INVENTORY; ++ i)
    if (O_KEY == gamedata->inventory[i])  havekey = TRUE;
    if (havekey)
    {
      switch (object)
      {
        case O_COTTAGE:
          puts("You find a switch inside.");
          gamedata->objects[O_SWITCH] = gamedata->room;
          return GAME_CONTINUES;
        case O_DOOR:
          beep(BEEP_INFO);
          puts("It opens revealing a laboratory.");
          gamedata->map[20 - 1] |= EXIT_WEST;
          return GAME_CONTINUES;
      }
    }
  cannotdothat();
return GAME_CONTINUES;
}

char turn(char object)
{
  switch (object)
  {
    case O_SWITCH:
      beep(BEEP_OK);
      puts("Thud.");
      gamedata->map[22 - 1] |= EXIT_WEST;
      break;
    default:
      cannotdothat();
      break;
  }
return GAME_CONTINUES;
}

char hit(char object)
{
  switch (object)
  {
    case O_SQUIRREL:
      puts("You kill the squirrel.");    // If we hit the squirrel, we kill it.
      return GAME_LOST;
    case O_SCIENTIST:
      gamedata->seenscientist = FALSE;         // Clear the seen scientist flag
      gamedata->objects[O_SCIENTIST] = -1;        // Remove scientist from game
      beep(BEEP_QUESTION);
      putstring("He will not bother you again.");
      break;
    default:
      cannotdothat();
      break;
  }
return GAME_CONTINUES;
}

char kill(void)
{
  putstring("No strong violence please. This is a family game.");
  return GAME_CONTINUES;
}

char pour(char object)
{
  switch (object)
  {
    case O_ELIXIR:
    /* The game object is to mix the potion with salt water. This can only
     be done by pouring it into the sea. */
      switch (gamedata->room)
      {
        case 6:
          puts("The potion has been neutralised.");
          return GAME_WON;
        default:
          puts("The potion poisons the land.");
          return GAME_LOST;
      }
      break;
    case O_COKE:
      puts("The bottle is empty.");
      break;
    default:
      cannotdothat();
      break;
  }
return GAME_CONTINUES;
}

char drink(char object)
{
  switch (object)
  {
    case O_ELIXIR:
      /* Can only drink the elixir, and we lose if we do it */
      puts("You die instantly.");
      return GAME_LOST;
    case O_COKE:
      /* It's empty */
      return pour(O_COKE);
      break;
    default:
      cannotdothat();
      return GAME_CONTINUES;
  }
}

char throw(char object)
{
  if ((object <= O_LAST_STONE || O_COKE == object) && 9 == gamedata->room)
  {
    return hit(O_SQUIRREL);
  }
  else
  {
    return drop(object);
  }
}

char look(char object)
{
  switch (object)
  {
    case O_STONE_MISSION:
      return read(object);
    case O_STONE_PILL:
      if (!gamedata->pilleaten && -1 == gamedata->objects[O_PILL])
        return take(object);
        /* Else fall through */
    default:
      nothing();
      return GAME_CONTINUES;
  }
}

char drop(char object)
{
  switch (object)
  {
    case O_COKE:
                        /* Can only recycle the Cola bottle once we've taken it */
      switch (gamedata->room)
      {
        case 12:
          if (dropinventory(O_COKE))
          {
            putstring("You recycle the bottle. Mother Nature thanks you!");
            gamedata->objects[O_COKE] = -1; /* Remove from game */
            break;
          }
          /* else fall through */
        case 28: case 29: /* Dump */
          putstring("Throwing away a perfectly recyclable bottle? No way!");
          break;
        /* Fall through for small build */
        case 1:  case 2:  case 3:
        case 7:  case 8:  case 9:
        case 13: case 14: case 15:
        case 30: /* Forest */
          puts("Someone might get hurt!");
          break;
        /* Fall through for small build */
        default:
          puts("Please recycle the bottle.");
          break;
      }
    break;
    case O_ELIXIR:
      if (gamedata->bothbottles)
      {
        putstring("You dare not drop the potion in its current state!");
        break;
      }
      /* else fall through */
    default:
      if (dropinventory(object))  puts("Dropped.");
  }
return GAME_CONTINUES;
}

char license(void)
{
  putstring("This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, version 2. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.");
return GAME_CONTINUES;
}

char quit(void)
{
  putstring("You leave for your home country. In the distance, you can hear the evil scientist laughing mercilessly as he unleashes his horrible magical potion on the land.");
return GAME_LOST;
}

//-----------------------------------------------------------------------------
//  GETINPUT - Get Command Line and Parse into Verb and Noun                  |
//-----------------------------------------------------------------------------
char getinput(void)
{
  while (1)
  {
    char i;
    const struct verb_s *verb;
    puts("Your turn? ");
    gets(input);
    input[1023] = 0;
//    input[strlen(input) - 1] = 0;
//    PUTCHAR('\n');
    verb = verbs;
    for (i = VERBS; i; -- i, ++ verb)
    {
      char n;
      char *p;
      n = verb->len;                                             // Verb length
      p = input + n;                              // First character after verb
      if (!strncmp(input, verb->s, n) && (!*p || ' ' == *p))
      {
        if (' ' == *p)
        {
          object_string = p + 1;
        }
        else
        {
          object_string = NULL;
        }
        return verb->verbnum;
      }
    }
    puts("I do not understand.");
  }
}

void printobject(char objnum)
{
const char *s;
  switch (objnum)
  {
    case O_STONE_PILL:
    case O_STONE_1:
    case O_STONE_2:
    case O_STONE_3:
    case O_STONE_MISSION:
    case O_STONE_5:
    case O_STONE_6:
    case O_STONE_7:
    case O_STONE_8:
      s = "A stone.";
      break;
    case O_PILL:
      s = "A pill.";
      break;
    case O_SQUIRREL:
      s = "A squirrel.";
      break;
    case O_COIN:
      s = "A gold coin.";
      break;
    case O_COTTAGE:
      s = "A cottage.";
      break;
    case O_SWITCH:
      s = "A switch.";
      break;
    case O_HUBCAP:
      s = "A hub cap.";
      break;
    case O_KEY:
      s = "A key.";
      break;
    case O_SCIENTIST:
      s = "A scientist.";
      break;
    case O_ELIXIR:
      s = "A potion.";
      break;
    case O_BOOK:
      s = "A book.";
      break;
    case O_DOOR :
      s = "A door.";
      break;
    case O_COKE:
      s = "Empty Coke bottle.";
      break; 
  }
  puts(s);
}

BOOL findobject(char _object, BOOL checkinventory)
{
char object;
char i;
char *inventory;
  object = _object;
  if (gamedata->objects[object] == gamedata->room)    // Is object in the room?
    return TRUE;        
  if (checkinventory)                        // Is the object in the inventory?
  {
    inventory = gamedata->inventory;
    for (i = INVENTORY; i; -- i, ++ inventory)
    if (*inventory == object)  return TRUE;
  }
return FALSE;
}

static struct objinfo_s
{
const char *noun;
char objnum;
} objinfo[NOUNS] =
{
  { "stone",    -1 },
  { "rock",     -1 },
  { "pill",     O_PILL },
  { "squirrel", O_SQUIRREL },
  { "coin",     O_COIN },
  { "gold coin",O_COIN },
  { "money",    O_COIN },
  { "cottage",  O_COTTAGE },
  { "house",    O_COTTAGE },
  { "door",     O_DOOR },
  { "switch",   O_SWITCH },
  { "hub cap",  O_HUBCAP },
  { "key",      O_KEY },
  { "door key", O_KEY },
  { "scientist",O_SCIENTIST },
  { "madman",   O_SCIENTIST },
  { "potion",   O_ELIXIR },
  { "book",     O_BOOK },
  { "bottle",   O_COKE },
  { "empty bottle",O_COKE },
  { "coke",     O_COKE },
  { "coca cola",O_COKE }
};

char objectfromword(const char *s, BOOL checkinventory)
{
char i, j;
struct objinfo_s *obj;
char objnum;
  j = -1;
  obj = objinfo;
  for (i = 0; i < NOUNS; ++ i, ++ obj)
  {
    if (!strcmp(s, obj->noun))
    {
      j = i;
      break;
    }
  }
  switch ((char) j)
  {
  case  -1:
    return -1;                                        // Not found in word list
  case 0:
  case 1:
    for (i = 0; i < 9; ++ i)                                // Stones (obj 0-8)
    {
      if (findobject(i, checkinventory))  return i;
    }
    break;
  default:
    objnum = obj->objnum;
    if (findobject(objnum, checkinventory))  return objnum;
    break;
  }
return -1;
}

BOOL addinventory(char object)
{
char i;
char *inventory;
  inventory = gamedata->inventory;
  for (i = INVENTORY; i; -- i, ++ inventory)
    switch ((char) *inventory)
    {
      case  -1:
        *inventory = object;
        gamedata->objects[object] = -1;
        return TRUE;
    }
  puts("It is too heavy.");
return FALSE;
}

BOOL dropinventory(char object)
{
char i;
char *inventory;
  inventory = gamedata->inventory;
  for (i = INVENTORY; i; -- i, ++ inventory)
    if ((char) object == *inventory)
    {
      *inventory = -1;
      gamedata->objects[object] = gamedata->room;
      return TRUE;
    }
  puts("You are not carrying that.");
return FALSE;
}

BOOL movable(char object)
{
  switch (object)
  {
    case O_PILL:
    case O_COIN:
    case O_HUBCAP:
    case O_KEY:
    case O_SCIENTIST:
    case O_ELIXIR:
    case O_BOOK:
    case O_COKE:
      return TRUE;
    default:
      return FALSE;
  }
}
//-----------------------------------------------------------------------------
// Putstring - Formats Text to Line and Outputs                               |
//-----------------------------------------------------------------------------
void putstring(const char *s)
{
static char tmpbuf[SCREENWIDTH];
size_t len;
  len = strlen(s);
  while (len >= SCREENWIDTH - 1)
  {
    const char *p;
    char *dest;
    const char *lastspace;
    size_t spaceleft;
    size_t width;
    p = s;
    dest = tmpbuf;
    spaceleft = SCREENWIDTH - 1;
    while (spaceleft && '\n' != *p)          // Loop through one line, locating 
//    while ((Graphics_string_width(main_module.m_gfx, tmpbuf) <160) && '\n' != *p) 
    {                                        // the first newline or last space
      switch (*p)
      {
        case ' ': lastspace = p;
      }
      *dest = *p;
      ++ dest;
      ++ p;
      -- spaceleft;
    }
    if ('\n' != *p)
    {
      p = lastspace;                                 // Terminate at last space
      width = p - s - 1;
      tmpbuf[width + 1] = 0;
    }
    else
    {
      *dest = 0;                                  // Terminate at found newline
      width = p - s;
    }
    puts(tmpbuf);                                        // Print the substring
    s = p + 1;                                               // Advance pointer
    len -= width;                                           // Decrease counter
  }
  puts(s);
}
//-----------------------------------------------------------------------------
// Perform                                                                    |
//-----------------------------------------------------------------------------
char perform(char verb)
{
char object;
  switch (verb)                          // Verbs that do not require an object
  {
    case V_N:
    case V_S:
    case V_E:
    case V_W:
      return move(verb);
    case V_I:
      return inventory();
    case V_LICENSE:
      return license();
    case V_QUIT:
      return quit();
  }                                            // Other words require an object
  object = object_string ? objectfromword(object_string, verb != V_TAKE) : -1;
  if (-1 == object)
  {
    if (V_LOOK == verb)
    {
      printroom();
    }
    else
    {
    puts("I do not see that here.");
    }
    return GAME_CONTINUES;
  }
  switch (verb)
  {
    case V_TAKE:
      return take((char) object);
    case V_LIFT:
      return lift((char) object);
    case V_READ:
      return read((char) object);
    case V_TALKTO:
      return talkto((char) object);
    case V_UNLOCK:
      return unlock((char) object);
    case V_TURN:
      return turn((char) object);
    case V_HIT:
      return hit((char) object);
    case V_KILL:
      return kill();
    case V_POUR:
      return pour((char) object);
    case V_DRINK:
      return drink((char) object);
    case V_THROW:
      return throw((char) object);
    case V_LOOK:
      return look((char) object);
    case V_DROP:
      return drop((char) object);
  }
  puts("Say what?");                                // This should never happen
return GAME_CONTINUES;
}

//-----------------------------------------------------------------------------
//  PRINTROOM - Room Description and Contents                                 |
//-----------------------------------------------------------------------------
void printroom(void)
{
char i;
BOOL seen;
const char *s;
char mapdata;
char *object;
  seen = FALSE;
  mapdata = gamedata->map[gamedata->room - 1];
  switch (gamedata->room)                             // Print room description
  {
    case 1:  case 2:  case 3:
    case 7:  case 8:  case 9:
    case 13: case 14: case 15:
    case 30:
      s = "You are in a forest. There are trees in every direction.";
      break;
    case 4: case 10:
    case 16: case 18:
      s = "You are standing on a big plain. You hear the faint hum of motors.";
      break;
    case 12:
      s = "You are standing on a big plain, next to a recycling station.";
      break;
    case 5: case 11: case 17:
    case 23: case 24:
      switch (gamedata->roadvisit)
      {
        case 0:
        case 1:
          ++ gamedata->roadvisit;
        case 3:
          s = "You are standing on an autobahn. Several cars are passing by.";
          break;
        case 2:
          s = "You are on an autobahn. A skateboarder gets a coke from a car.";
  //            "dr pass by, grabbing a bottle of Coca Cola from a nearby car.";
          ++ gamedata->roadvisit;
          gamedata->objects[O_COKE] = gamedata->room;
          break;
      }
      break;
    case 6:
      s = "You are on the seashore. The sea stretches to the north and east.";
      break;
    case 19: case 25:
      s = "You are inside the scientist's lab. It looks rather untidy.";
      break;
    case 20: case 21: case 26: case 27:
      s = "You are standing inside a big fortress. Dust and cobwebs everywhere.";
      break;
    case 22:
      if (!(char) (mapdata & EXIT_WEST))
      {
        s = "You are standing beside a big moat. The drawbridge is up.";
      }
      else
      {
        s = "You are standing beside a drawbridge leading to a big fortress.";
      }
      break;
    case 28: case 29:
      s = "You are standing on a dump. The stench permiates the air!";
      break;
    default:
      s = "Something is very wrong!";
      break;
  }
  putstring(s);
  object = gamedata->objects;                            // Print room contents
  for (i = 0; i < OBJECTS; ++ i, ++ object)
  {
    if (*object == gamedata->room)
    {
      if (!seen)
      {
        puts("You see:");
        seen = TRUE;
      }
      printobject(i);
      if (O_SCIENTIST == i && !gamedata->seenscientist)
      {
        puts("He says: \"What are you doing here?\"");
        gamedata->seenscientist = TRUE;
      }
    }
  }
  puts("Available exits:");                                      // Print exits
  if ((mapdata & EXIT_NORTH) != 0)
    puts("N");
  if ((mapdata & EXIT_SOUTH) != 0)
    puts("S");
  if ((mapdata & EXIT_EAST) != 0)
    puts("E");
  if ((mapdata & EXIT_WEST) != 0)
    puts("W");
//  PUTCHAR('\n');
}

//-----------------------------------------------------------------------------
// Init Con - Initializes Console                                             | 
//-----------------------------------------------------------------------------
int InitCon()
{
  DisplayGraphics_fill_screen(main_module.m_gfx, CLR_WHITE);
}

//-----------------------------------------------------------------------------
// Puts - Outputs Line of Text with Newline                                   |
//-----------------------------------------------------------------------------
int puts(char *Sbuf)
{
  DisplayGraphics_draw_text(main_module.m_gfx, Sbuf, 0, 90);
  Scroll();
  DisplayGraphics_show(main_module.m_gfx);                    // Update Display
}
//-----------------------------------------------------------------------------
// Clrline - Clears line and restores text color                              |
//-----------------------------------------------------------------------------
void Clrline(void)
{
  DisplayGraphics_set_color (main_module.m_gfx, CLR_WHITE);
  DisplayGraphics_fill_rect( main_module.m_gfx, 0, 90, 160, 9);
  DisplayGraphics_set_color (main_module.m_gfx, CLR_BLACK);
}

//-----------------------------------------------------------------------------
// Scroll - Scrolls screen upward 9 pixels                                    |
//-----------------------------------------------------------------------------
void Scroll(void)
{
  DisplayGraphics_scroll(main_module.m_gfx, 0, 0, 160, 99, 0, -9);   // Was -10
  Clrline();
}

//-----------------------------------------------------------------------------
// Get Char - Waits until a key is pressed, Returns key pressed               |
//-----------------------------------------------------------------------------
int getchar(void)
{
int key;
bool pressed = FALSE;
  while(!pressed)                                  // Loop until key is pressed
    {
    ptr_msg = cWinApp_get_message(main_module.m_process, 0, 1, MSG_USER);
    switch(ptr_msg->msgid)
      {
      case MSG_KEYDOWN:
        key = Message_get_key_param(ptr_msg)->scancode;     // Allows spec keys
        switch(key)
          {
          case KEY_ENTER:
            pressed = TRUE;
            break;
          case KEY_BACKSPACE:
            pressed = TRUE;
            break;
          case KEY_UP:
            pressed = TRUE;
            break;
          case KEY_DOWN:
            pressed = TRUE;
            break;
          case KEY_LEFT:
            pressed = TRUE;
            break;
          case KEY_RIGHT:
            pressed = TRUE;
            break;
          case KEY_SELECT:
            pressed = TRUE;
            break;
          case KEY_TAB:
            pressed = TRUE;
            break;
          case KEY_SHIFT:        // So SHIFT doesn't register as a key val of 0
            break;
          case KEY_CONTROL:                    // THE FIX (for Cybiko Extremes)
            break;
          default:                    // For the rest of the alpha-numeric keys
            pressed = TRUE;
            key = Message_get_key_param(ptr_msg)->ch;  // Allows SHIFT key vals
            break;
          }
          break;
        default:
        cWinApp_defproc(main_module.m_process, ptr_msg);   // Process Remaining
      }
      Message_delete(ptr_msg);                      // Delete processed message
    } 
return key;
}

//-----------------------------------------------------------------------------
// Get String                                                                 |
//-----------------------------------------------------------------------------
char *gets(char *tmp)
{
int key;
char a;
bool done = FALSE;       // (for(a=0; a<64; B[a++]=0); generates 2 extra bytes)
for(a=0; a<64; a++) tmp[a]=0;                            // Clear buffer string
a=0;
  while(!done)                                   // Loop Until Enter is Pressed
    {
    key = getchar();                             // Wait until a key is pressed
    switch(key)
      {
      case KEY_ENTER:
        tmp[a] = 0;
        done = TRUE;
        break;
      case KEY_BACKSPACE:
        if(a) tmp[--a]=0;                     // If length > 0 delete last char
        break;
      default:
        tmp[a]=(char)key;                         // Add the char and increment
        a++;                     // (B[a++]=(char)key; generates 4 extra bytes)
        break;
      }
    Clrline();
    DisplayGraphics_draw_text(main_module.m_gfx, tmp, 0, 90);
    DisplayGraphics_show(main_module.m_gfx);                  // Update Display
    }
  Scroll();
return tmp;
}
